<?php

/**
 * Plugin Name:       DexterGpt
 * Description:       DexterGpt - AI Writing Tool Plugin For Text Generation
 * Version:           1.0.0
 * Author:            DexterGpt
 * License:           GPL-2.0 or later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       dextergpt
 * Domain Path:       /languages
 */

// If this file is called directly, abort.
defined('WPINC') || exit;

include_once __DIR__ . '/utils.php';

if (!class_exists('DexterGpt')) {
    class DexterGpt
    {
        public $plugin_slug;
        public $plugin_path;
        public $version = '1.0.0';
        /**
         * @var \DexterGpt\APIClient|null
         */
        private $api_client = null;
        private $settings = null;
        private $log_file = __DIR__ . '/log.php';

        const SETTINGS_KEY = 'dextergpt_settings';
        const REST_VERSION = 1;
        const MB_ENCODING = 'UTF-8';

        const SCHEMA_TYPE_JSON = 'json';

        public function __construct()
        {

            set_exception_handler(function ($e) {
                $record = [
                    'type' => 'exceptionHandler',
                    'message' => $e->getMessage(),
                    'code' => $e->getCode(),
                    'backtrace' => []
                ];
                if (isset($_SERVER['REQUEST_URI'])) {
                    $replace = explode('wp-json', $_SERVER['REQUEST_URI'])[0];
                } else {
                    $replace = '';
                }
                foreach ($e->getTrace() as $item) {
                    if (isset($item['file'])) {
                        $file = $replace === '' ? $item['file'] : explode($replace, $item['file'])[1];
                    } else {
                        $file = 'unknown';
                    }
                    if (isset($item['line'])) {
                        $line = $item['line'];
                    } else {
                        $line = -1;
                    }
                    $record['backtrace'][] = [
                        $file,
                        $line
                    ];
                }
                $this->writeLog($record);
            });

            $this->plugin_slug = plugin_basename(__DIR__);
            $this->plugin_path = plugin_dir_path(__FILE__);

            if (is_admin()) {
                $this->adminPages();
                register_deactivation_hook(__FILE__, [$this, 'deactivate']);
            }

            if (wp_doing_ajax()) {
                add_action('wp_ajax_dextergpt-notify', [$this, 'ajaxWebhook']);
                add_action('wp_ajax_nopriv_dextergpt-notify', [$this, 'ajaxWebhook']);
            }

            add_action('rest_api_init', [$this, 'initRest']);
            add_filter('wp_kses_allowed_html', [$this, 'ksesAllowedHtml'], 10, 2);

            add_filter('the_content', [$this, 'restoreSchemaSection'], 20);
            add_action("wp_head", [$this, 'printJSONLD'], 20);
        }

        public function ksesAllowedHtml($allowed, $context)
        {
            if (!is_array($context) && ($context === 'post')) {
                $allowed['div']['itemscope'] = true;
                $allowed['div']['itemprop'] = true;
                $allowed['div']['itemtype'] = true;
                $allowed['h3']['itemprop'] = true;
                $allowed['a']['rel'] = true;
                $allowed['a']['target'] = true;
            }
            return $allowed;
        }

        public function initRest()
        {
            register_rest_route($this->getRestNamespace(), '/webhook', [
                'methods' => 'POST',
                'callback' => [$this, 'restWebhook'],
                'permission_callback' => '__return_true'
            ]);
            register_rest_route($this->getRestNamespace(), '/debug', [
                'methods' => 'POST',
                'callback' => [$this, 'restDebug'],
                'permission_callback' => '__return_true'
            ]);
            register_rest_route($this->getRestNamespace(), '/log', [
                'methods' => 'POST',
                'callback' => [$this, 'restLog'],
                'permission_callback' => '__return_true'
            ]);
        }

        public function getRestNamespace()
        {
            return 'dextergpt/v' . self::REST_VERSION;
        }

        /**
         * @param WP_REST_Request $request
         * @return WP_REST_Response|WP_Error
         */
        public function restLog($request)
        {
            $res = [];
            $post = $request->get_json_params();
            $settings = $this->getSettings();
            $client = $this->getAPIClient();
            if (isset($settings['secret'])
                && $client->checkSecter($post, $settings['secret'])
                && is_readable($this->log_file)
            ) {
                // @phpstan-ignore-next-line
                $res = json_decode('[' . str_replace("}\n", "},", trim(explode('?>', @file_get_contents($this->log_file))[1])) . ']', true);
            }
            return new WP_REST_Response($res, is_array($res) && count($res) > 0 ? 200 : 400);
        }

        /**
         * @param WP_REST_Request $request
         * @return WP_REST_Response|WP_Error
         */
        public function restDebug($request)
        {
            $res = [];
            $post = $request->get_json_params();
            $settings = $this->getSettings();
            $client = $this->getAPIClient();
            if (
                get_option('dextergpt_debug') === 'yes'
                && isset($settings['secret'])
                && isset($post['sign'])
                && $client->checkSecter($post, $settings['secret'])) {
                include_once __DIR__ . '/../../../wp-admin/includes/plugin.php';
                $plugins = [];
                if (function_exists('get_plugins')) {
                    foreach (get_plugins() as $pluginFile => $settings) {
                        $plugins[] = [
                            'name' => esc_html(explode('/', $pluginFile)[0]),
                            'version' => esc_html($settings['Version']),
                            'active' => is_plugin_active($pluginFile),
                        ];
                    }
                }
                $res = [
                    'engine' => [
                        'name' => 'wordpress',
                        'version' => get_bloginfo('version'),
                    ],
                    'php' => [
                        'version' => esc_html(PHP_VERSION),
                    ],
                    'webServer' => [
                        'name' => esc_html(isset($_SERVER['SERVER_SOFTWARE']) ? $_SERVER['SERVER_SOFTWARE'] : 'Unknown'),
                    ],
                    'plugins' => $plugins,
                ];
            }

            return new WP_REST_Response($res, count($res) > 0 ? 200 : 400);
        }

        /**
         * @param WP_REST_Request $request
         * @return WP_REST_Response|WP_Error
         */
        public function restWebhook($request)
        {

            $rs = $request->has_valid_params();
            if (is_wp_error($rs)) {
                $this->writeLog([
                    'type' => 'restWebhookValidation',
                    'data' => $rs->get_error_messages(),
                ]);
                return new WP_REST_Response(['message' => $rs->get_error_messages()], 400);
            }

            // WP 4.9 compatibility, do not edit
            $contentType = $request->get_content_type();
            if (!is_array($contentType) || !isset($contentType['value']) || $contentType['value'] !== 'application/json') {
                $error = 'Wrong Content-Type: ' . (isset($contentType['value']) ? $contentType['value'] : '');

                $this->writeLog([
                    'type' => 'restWebhookContentType',
                    'message' => $error,
                ]);
                return new WP_REST_Response(['message' => $error], 400);
            }

            $ret = $this->webhook($request->get_json_params());
            return is_array($ret)
                ? new WP_REST_Response($ret)
                : new WP_REST_Response([
                    'message' => 'You do not have permission to access this resource.',
                ], 400);

        }

      

        public function getSettings()
        {
            if (is_null($this->settings)) {
                $this->settings = get_option(self::SETTINGS_KEY, []);
                if (!is_array($this->settings)) {
                    $this->settings = [];
                }
            }
            return $this->settings;
        }

        public function setSettings($data)
        {
            $modified = false;
            foreach ($data as $k => $v) {
                if (is_null($v)) {
                    if (isset($this->settings[$k])) {
                        unset($this->settings[$k]);
                        $modified = true;
                    }
                } elseif (!isset($this->settings[$k]) || ($this->settings[$k] !== $v)) {
                    $this->settings[$k] = $v;
                    $modified = true;
                }
            }

            if ($modified) {
                if (count($this->settings) > 0) {
                    update_option(self::SETTINGS_KEY, $this->settings);
                } else {
                    delete_option(self::SETTINGS_KEY);
                }
            }
        }

        public function deleteSettings()
        {
            $this->settings = [];
            delete_option(self::SETTINGS_KEY);
        }

        /**
         * @return bool
         */
        public function isConnected()
        {
            $settings = $this->getSettings();
            return !empty($settings['api_key']);
        }

        public function connect($params)
        {
            return $this->getAPIClient()->connect($params);
        }

        public function disconnect()
        {
            return $this->getAPIClient()->disconnect();
        }

        private function getWebhookData()
        {
            $post = file_get_contents('php://input');
            if (!empty($post)) {
                $_data = @json_decode($post, true);
                if (is_array($_data)) {
                    return $_data;
                }
            }
            return [];
        }

        /**
         * @param array<string, string> $post
         * @return array<string, string>|null
         */
        private function webhook($post)
        {
            $settings = $this->getSettings();

            if (isset($settings['secret'])) {
                $client = $this->getAPIClient();
                if ($client->checkSecter($post, $settings['secret'])) {
                    $action = isset($post['action']) ? sanitize_text_field($post['action']) : '';
                    $ret = [
                        'result' => 0,
                        'version' => $this->version,
                    ];

                    if ($action === 'post') {
                        $rs = $this->publishPost($settings['user_id'], $post);
                    } elseif ($action === 'image_upload') {
                        $img = isset($post['img']) ? sanitize_text_field($post['img']) : '';
                        $name = isset($post['name']) ? sanitize_text_field($post['name']) : '';
                        $type = isset($post['type']) ? sanitize_text_field($post['type']) : '';

                        if ($type === 'base64') {
                            $url = $this->imageUploadBase64($img, $name);
                            $rs = [
                                'result' => 1,
                                'image' =>  $url,
                            ];
                        } else if ($type === 'url') {
                            $url = $this->imageUploadUrl($img, $name);
                            $rs = [
                                'result' => 1,
                                'image' =>  $url,
                            ];
                        } else {
                            $rs = [
                                'error' => 'image type not supported ',
                                'types' => ['base64', 'url']
                            ];
                        }
                    }
                    elseif (($action === 'connect') && isset($post['api_key'])) {
                        $this->setSettings([
                            'api_key' => sanitize_text_field($post['api_key']),
                        ]);
                        $rs = [
                            'result' => 1,
                            'categories' => $this->getCategories()
                        ];
                    } elseif ($action === 'disconnect') {
                        $this->deleteSettings();
                        $rs = [
                            'result' => 1
                        ];
                    } elseif ($action === 'get_categories') {
                        $rs = [
                            'result' => 1,
                            'categories' => $this->getCategories()
                        ];
                    } else {
                        $rs = [
                            'error' => 'Plugin does not support this feature'
                        ];
                    }

                    return array_merge($ret, $rs);
                }
            }
            return null;
        }

        public function ajaxWebhook()
        {
            $ret = $this->webhook($this->getWebhookData());
            if (is_array($ret)) {
                wp_send_json($ret);
            }
            exit();
        }

        /**
         * @param $content
         * @return string
         */
        private function clearSchemaSection($content)
        {
            if (preg_match('#<section itemscope itemprop="mainEntity" itemtype="https://schema.org/FAQPage">(.*?)</section>#s', $content, $matches)) {
                $html = $matches[0];
                $html = str_replace('itemscope itemprop="mainEntity" itemtype="https://schema.org/FAQPage"', 'class="schema-section"', $html);
                $items = array(' itemscope', ' itemprop="mainEntity"', ' itemprop="text"', ' itemprop="name"', ' itemprop="acceptedAnswer"', ' itemtype="https://schema.org/Question"', ' itemtype="https://schema.org/Answer"');
                $html = str_replace($items, "", $html);
                $content = preg_replace('#<section itemscope itemprop="mainEntity" itemtype="https://schema.org/FAQPage">(.*?)</section>#s', $html, $content);
            }

            return $content;
        }

        /**
         * @return bool
         */
        private function isJSONSchema()
        {
            return get_option('sw_shema_type') === self::SCHEMA_TYPE_JSON;
        }

        /**
         * @return bool
         */
        private function isMicrodataSchema()
        {
            return !$this->isJSONSchema();
        }

        public function printJSONLD()
        {
            if (!is_single() || !$this->isJSONSchema()) {
                return;
            }
            // WP 4.9 compatibility, do not edit
            $post = get_post(get_queried_object_id());
            if ($post instanceof WP_Post) {
                $content = $post->post_content;
            } else {
                return;
            }
            $qa = $this->qaList($content);
            if (!is_array($qa) || !isset($qa[1])) {
                return;
            }
            $questions = $qa[0];
            $answers = $qa[1];
            $count = count($questions);
            $items = '';
            for ($i = 0; $i < $count; $i++) {
                if (isset($answers[$i]) && isset($questions[$i])) {
                    $items .= '{'
                        . '"@type": "Question",'
                        . '"name": "' . esc_html($questions[$i]) . '",'
                        . '"acceptedAnswer": {'
                        . '"@type": "Answer",'
                        . '"text": "' . esc_html($answers[$i]) . '"'
                        . '}'
                        . '}';
                    if ($i != $count - 1) {
                        $items .= ',';
                    }
                }
            }
            echo '<script type="application/ld+json">'
                . '{'
                . '"@context": "https://schema.org",'
                . '"@type": "FAQPage",'
                . '"mainEntity": [' . $items . ']'
                . '}'
                . '</script>';
        }

        private function qaList($html)
        {
            if (preg_match('#<section class="schema-section">(.*?)</section>#s', $html, $matches)) {
                $title = '';
                $fhtml = $matches[1];
                if (preg_match('#<h2>(.*?)</h2>#s', $fhtml, $titles)) {
                    $title = $titles[1];
                    $fhtml = str_replace($titles[0], '', $fhtml);
                }
                $fhtml = strip_tags($fhtml, "<h3><p><b>");
                preg_match_all('#<h3>(.*?)</h3>#s', $fhtml, $questions);
                $questions = isset($questions[1]) ? $questions[1] : [];
                $answers = preg_split('#<h3>(.*?)</h3>#s', $fhtml);
                if (is_array($answers)) {
                    array_shift($answers);
                    foreach ($answers as $idx => $answer) {
                        $answers[$idx] = trim(str_replace(PHP_EOL, "", strip_tags($answer)));
                    }
                }
                return [$questions, $answers, $title];
            }

            return false;
        }

        public function restoreSchemaSection($content)
        {
            if (!is_single() || !$this->isMicrodataSchema()) {
                return $content;
            }
            $qa = $this->qaList($content);
            if (!isset($qa[2]) || !isset($qa[0][0]) || !isset($qa[1][0])) {
                return $content;
            }
            $questions = $qa[0];
            $answers = $qa[1];
            $title = $qa[2];
            $count = count($questions);

            $out = '<section itemscope itemtype="https://schema.org/FAQPage">';
            $out .= '<h2>' . $title . '</h2>';
            for ($i = 0; $i < $count; $i++) {
                if (isset($answers[$i]) && isset($questions[$i])) {
                    $out .= '<div itemscope itemprop="mainEntity" itemtype="https://schema.org/Question">'
                        . '<h3 itemprop="name">' . $questions[$i] . '</h3>'
                        . '<div itemscope itemprop="acceptedAnswer" itemtype="https://schema.org/Answer">'
                        . '<div itemprop="text">' . $answers[$i] . '</div>'
                        . '</div>'
                        . '</div>';
                }
            }
            $out .= '</section>';

            $content = preg_replace('#<section class="schema-section">(.*?)</section>#s', $out, $content);

            return $content;
        }

        private function imageUploadUrl($image_url, $filename)
        {
            $url_components = parse_url($image_url);
            $has_query_params = isset($url_components['query']) && !empty($url_components['query']);
            $image_extension_pattern = '/\.(jpg|jpeg|png|gif|webp)$/i';
            $has_valid_extension = preg_match($image_extension_pattern, $image_url);
            if ($has_query_params ||  !$has_valid_extension) {
                $newBase64 = $this->get_image_base64_from_url($image_url);
                $url =  $this->imageUploadBase64($newBase64, $filename);
                error_log('GÖRSEL URL DEN BASE64 TİPİNDE YÜKLENDİ İD : ' . $url);
                return $url;
            }


            $api = $this->getAPIClient();
            $file = $api->loadImage($image_url, $filename);
            if ($file) {
                include_once(ABSPATH . 'wp-admin/includes/image.php');
                include_once(ABSPATH . 'wp-admin/includes/file.php');
                include_once(ABSPATH . 'wp-admin/includes/media.php');
                $id = media_handle_sideload($file, 0);
                $api->deleteImage($file);
                error_log('GÖRSEL URL TİPİNDE YÜKLENDİ İD : ' . $id);
                return [
                    'url' => wp_get_attachment_url($id),
                    'id' => $id
                ];
            }
        }

        private function imageUploadBase64($base64_image, $filename)
        {
            $base64_string = preg_replace('/^data:image\/[a-zA-Z]+;base64,/', '', $base64_image);
            $image_data = base64_decode($base64_string);
            if ($image_data === false) {
                return false;
            }
            $upload_dir = wp_upload_dir();
            $secret = md5(time() . mt_rand());
            $file_path = $upload_dir['path'] . '/' . sanitize_file_name($filename) . '-' . $secret . '.jpg';
            if (file_put_contents($file_path, $image_data) === false) {
                return false;
            }
            $filetype = wp_check_filetype($file_path);
            if (!$filetype['type']) {
                return false;
            }
            include_once(ABSPATH . 'wp-admin/includes/image.php');
            include_once(ABSPATH . 'wp-admin/includes/file.php');
            include_once(ABSPATH . 'wp-admin/includes/media.php');
            $attachment = array(
                'guid' => $upload_dir['url'] . '/' . basename($file_path),
                'post_mime_type' => $filetype['type'],
                'post_title' => preg_replace('/\.[^.]+$/', '', basename($file_path)),
                'post_content' => '',
                'post_status' => 'inherit'
            );
            $attachment_id = wp_insert_attachment($attachment, $file_path);
            if (!is_wp_error($attachment_id)) {
                $attachment_data = wp_generate_attachment_metadata($attachment_id, $file_path);
                wp_update_attachment_metadata($attachment_id, $attachment_data);
                return [
                    'url' => wp_get_attachment_url($attachment_id),
                    'id' => $attachment_id
                ];
            }

            return false;
        }


        function get_image_base64_from_url($image_url)
        {
            $image_data =  wp_remote_get($image_url, array(
                'headers' => array(
                    'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/85.0.4183.121 Safari/537.36',
                    'Accept' => 'image/webp,image/apng,image/*,*/*;q=0.8',
                    'Referer' => home_url(),
                ),
            ));
            if (is_wp_error($image_data)) {
                return false;
            }
            $response_code = wp_remote_retrieve_response_code($image_data);
            if ($response_code != 200) {
                return false;
            }
            $image_body = wp_remote_retrieve_body($image_data);
            $mime_type = wp_remote_retrieve_header($image_data, 'content-type');
            if (!$mime_type) {
                error_log('MIME tipi belirlenemedi');
                return false;
            }
            error_log(time(). ' MIME tipi belirlendi' . $mime_type);
            $base64_image = 'data:' . $mime_type . ';base64,' . base64_encode($image_body);
            return $base64_image;
        }




        private function downloadImages(&$data, $post_id, $featured_image)
        {
            $html = $data['html'];

            if (preg_match_all('/<img .*src="([^">]+)"[^>]*>/uU', $html, $matches)) {
                $data['images'] = [];
                include_once(ABSPATH . 'wp-admin/includes/image.php');
                include_once(ABSPATH . 'wp-admin/includes/file.php');
                include_once(ABSPATH . 'wp-admin/includes/media.php');
                $api = $this->getAPIClient();
                $images = [];
                foreach ($matches[1] as $i => $path) {
                    if ($api->checkImageUrl($path)) {
                        $alt = '';
                        if (preg_match('/ alt="([^"]*)"/u', $matches[0][$i], $_matches)) {
                            $alt = html_entity_decode($_matches[1], ENT_COMPAT | ENT_HTML401, self::MB_ENCODING);
                        }

                        $attachment_id = 0;
                        if (in_array($path, $images)) {
                            $attachment_id = array_search($path, $images);
                        } elseif ($file = $api->loadImage($path, $alt)) {
                            $id = media_handle_sideload($file, $post_id);
                            $api->deleteImage($file);

                            if (is_wp_error($id)) {
                                $this->writeLog([
                                    'type' => 'downloadImagesWpError',
                                    'data' => $id->get_error_messages(),
                                ]);
                            } else {
                                $attachment_id = $id;
                                $images[$id] = $path;

                                if (strlen($alt) > 0) {
                                    update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
                                }
                            }
                        } else {
                            $this->writeLog([
                                'type' => 'downloadImagesWpError',
                                'path' => $path,
                                'data' => $api->error,
                            ]);
                        }

                        if ($attachment_id > 0) {
                            if ($featured_image) {
                                set_post_thumbnail($post_id, $attachment_id);
                                $featured_image = false;

                                $html = str_replace($matches[0][$i], '', $html);
                            } else {
                                $data['images'][wp_get_attachment_url($attachment_id)] = $attachment_id;
                                $html = str_replace($matches[0][$i],
                                    get_image_tag($attachment_id, $alt, $alt, 'center', 'large'),
                                    $html
                                );
                            }
                        }
                    }
                }

                if (count($images) > 0) {
                    wp_update_post([
                        'ID' => $post_id,
                        'post_content' => wp_slash($html)
                    ]);
                }

                $data['html'] = $html;
            }
        }

        /**
         * @param string $category
         * @return array<int>
         */
        private function getPostCategory($category)
        {
            if (strlen($category) > 0) {
                $categories = $this->getCategories();
                $ids = array_map('intval', explode(',', $category));

                $result = [];
                foreach ($categories as $cat) {
                    if (in_array($cat['id'], $ids)) {
                        $result[] = (int)$cat['id'];
                    }
                }

                if (count($result) > 0) {
                    return $result;
                }
            }

            return [0];
        }

        private function publishPost($user_id, $data)
        {
            $maxExecutionTime = (int)ini_get('max_execution_time');
            @set_time_limit(120);

            $post_status = (isset($data['publish']) && intval($data['publish']) === 1) ? 'publish' : 'draft';
            $post_time = time();
            date_default_timezone_set('Europe/Istanbul');

            if (($post_status === 'publish') && isset($data['post_time'])) {
               // $new_post_time = intval($data['post_time']);
                $new_post_time = strtotime($data['post_time']); // Unix timestamp'e çeviriyoruz

                if ($new_post_time > 0) {
                    if ($new_post_time > $post_time) {
                        $post_status = 'future';
                    }
                    $post_time = $new_post_time;
                }
            }
            $content = wp_kses_post(
                $this->clearSchemaSection(
                    str_replace('itemscope="" itemprop=', 'itemscope itemprop=', $data['html'])
                )
            );
            $data['html'] = $content;
            $new_post = [
                'post_title' => sanitize_text_field($data['title']),
                'post_content' => $content,
                'post_status' => $post_status,
                'post_date' => date('Y-m-d H:i:s', $post_time),
                'post_author' => $user_id,
                'post_type' => 'post',
                'post_category' => $this->getPostCategory(isset($data['category']) ? sanitize_text_field($data['category']) : ''),
                'post_excerpt' => isset($data['excerpt']) && (int)$data['excerpt'] === 1
                    ? (isset($data['description']) ? sanitize_text_field($data['description']) : '')
                    : '',
            ];

            if (isset($data['post_slug'])) {
                $slug = sanitize_title($data['post_slug']);
                if (strlen($slug) > 0) {
                    $new_post['post_name'] = $slug;
                }
            }

            $post_id = wp_insert_post($new_post);

            if (is_wp_error($post_id)) {
                return [
                    'result' => 0,
                    'error' => $post_id->get_error_message(),
                ];
            }

            if (!empty($data['tags']) && is_array($data['tags'])) {
                $rs = wp_set_post_tags($post_id, $data['tags']);
                if (is_wp_error($rs)) {
                    $this->writeLog([
                        'type' => 'setPostTags',
                        'id' => $post_id,
                        'data' => $rs->get_error_message(),
                        'tags' => $data['tags'],
                    ]);
                }
            }

          //  $this->downloadImages($data, $post_id, isset($data['featured_image']));

            include_once(__DIR__ . '/classes/post-meta.php');
            $pm = new \DexterGpt\PostMeta($post_id);
            $pm->set($data);

            if(isset($data['featured_image'])){
                $featured_image_id = $data['featured_image'];
                set_post_thumbnail($post_id, $featured_image_id);
               }

            @set_time_limit($maxExecutionTime);

            return [
                'result' => 1,
                'post_id' => $post_id,
                'url' => wp_get_shortlink($post_id),
            ];
        }

        /**
         * @return array<array<string, int|string>>
         */
        public function getCategories()
        {
            $categories = get_categories([
                'hide_empty' => 0
            ]);

            $array = [];
            foreach ($categories as $category) {
                /** @var WP_Term $category */
                $array[] = [
                    'id' => (int)$category->term_id,
                    'name' => $category->name,
                    'parent' => (int)$category->parent
                ];
            }

            return $array;
        }

        /**
         * @return \DexterGpt\APIClient
         */
        private function getAPIClient()
        {
            if (is_null($this->api_client)) {
                require_once($this->plugin_path . 'classes/api-client.php');
                $this->api_client = new \DexterGpt\APIClient($this);
            }
            return $this->api_client;
        }

        public function adminPages()
        {
            include_once($this->plugin_path . 'classes/settings-form.php');

            $form = new \DexterGpt\SettingsForm($this);
            $form->init();
        }

        public function deactivate()
        {
            if ($this->isConnected()) {
                $this->disconnect();
            }
            $this->deleteSettings();
        }

        private function writeLog($s)
        {
            if (!is_array($s)) {
                $s = [
                    'source' => $s,
                ];
            }
            $s['now'] = date('Y-m-d H:i:s');
            if (!file_exists($this->log_file)) {
                @file_put_contents($this->log_file, "<?php die(); ?>\n\n");
            }
            @file_put_contents($this->log_file, dextergpt_json_encode_unescaped($s) . "\n", FILE_APPEND);
        }

        /**
         * @static
         * @return DexterGpt
         */
        public static function i()
        {
            return new DexterGpt();
        }
    }

    DexterGpt::i();
}



function init(){
    $plugin_rel_path = basename( dirname( __FILE__ ) ) . '/languages';
	load_plugin_textdomain( 'dextergpt', false, $plugin_rel_path );
}

add_action('plugins_loaded', 'init');

function active_plugin_proc() {
    set_transient('dextergpt_eklenti_yonlendirme', true, 30); 
}
register_activation_hook(__FILE__, 'active_plugin_proc');
function redirect_plugin_settings() {
    if (get_transient('dextergpt_eklenti_yonlendirme') && current_user_can('manage_options')) {
        delete_transient('dextergpt_eklenti_yonlendirme');
        wp_safe_redirect(admin_url('admin.php?page=dextergpt-setting'));
        exit;
    }
}
add_action('admin_init', 'redirect_plugin_settings');