<?php

/**
 * Class  TIDN_Helpers for working with the 2index.ninja API
 */
class TIDN_Helpers
{
    /**
     * @return string
     */
    public static function getApiKey(): string
    {
        return get_option('TIDN_api_key', '');
    }

    /**
     * @param string $apiKey
     * @return void
     */
    public static function setApiKey(string $apiKey): void
    {
        update_option('TIDN_api_key', sanitize_text_field($apiKey));
    }

    /**
     * @return bool
     */
    public static function isApiKeyRegistered(string $apiKey): bool
    {
        $api = new  TIDN_Api($apiKey);
        $response = $api->account();
        if (self::isQueryError($response)) {
            self::removeApiKey();
            return false;
        }
        return self::isQuerySuccess($response);
    }

    /**
     * @return bool
     */
    public static function isApiKeyVerified(string $apiKey): bool
    {
        $api = new  TIDN_Api($apiKey);
        $response = $api->account();
        return $response['account']['email_verified'] ?? false;
    }

    /**
     * @return void
     */
    public static function removeApiKey(): void
    {
        delete_option('TIDN_api_key');
    }

    /**
     * @param $response
     * @return bool
     */
    public static function setApiKeyFromResponse($response): bool
    {
        if (!self::isResponseValid($response)) {
            return false;
        }
        if (self::isQuerySuccess($response)) {
            if (isset($response['account']['api_key'])) {
                self::setApiKey($response['account']['api_key']);
                return true;
            }
        }
        return false;
    }

    /**
     * @param $response
     * @return bool
     */
    public static function isQuerySuccess($response): bool
    {
        if (!self::isResponseValid($response)) {
            return false;
        }
        return $response['success'] == 1;
    }

    /**
     * @param $response
     * @return bool
     */
    public static function isQueryError($response): bool
    {
        if (!self::isResponseValid($response)) {
            return true;
        }
        return $response['success'] == 0;
    }

    /**
     * @param $response
     * @return string|bool
     */
    public static function getQueryMessage($response)
    {
        if (!self::isResponseValid($response)) {
            return "Invalid JSON response";
        }
        if (isset($response['message'])) {
            return $response['message'];
        }
        return false;
    }

    /**
     * @param $response
     * @return string|bool
     */
    public static function getQueryErrorMessage($response)
    {
        $message = self::getQueryMessage($response);
        if (!$message) {
            $message = self::getFirstError($response);
        }
        if (!$message) {
            $message = self::getUniversalErrorMessage();
        }
        return $message;
    }

    /**
     * @param $response
     * @return string|false
     */
    public static function getInvalidLinksText($response)
    {
        if (!self::isResponseValid($response)) {
            return false;
        }
        $invalidLinksText = false;
        if (isset($response['invalid_links']) && is_array($response['invalid_links'])) {
            $invalidLinks = $response['invalid_links'];
            $invalidLinksCount = count($invalidLinks);
            $firstThreeInvalidLinks = array_slice($response['invalid_links'], -3);
            $restInvalidLinksCount = $invalidLinksCount - 3;

            if ($invalidLinksCount > 0) {
                $linkMisc = ($invalidLinksCount == 1) ? "link" : "links";
                $invalidLinksText = "You sent $invalidLinksCount invalid $linkMisc<br>";

                $invalidLinksText .= "<ul>";
                foreach ($firstThreeInvalidLinks as $invalidLink) {
                    $invalidLinksText .= "<li>$invalidLink</li>";
                }
                $invalidLinksText .= "</ul>";
                $invalidLinksText .= ($restInvalidLinksCount > 0 ? "... and $restInvalidLinksCount more" : "");
            }
        }
        return $invalidLinksText;
    }

    /**
     * @param $response
     * @return mixed|string
     */
    public static function getFirstError($response): string
    {
        if (!self::isResponseValid($response)) {
            return "Invalid JSON response";
        }

        if (isset($response['errors']) && is_array($response['errors'])) {
            foreach ($response['errors'] as $fieldErrors) {
                if (is_array($fieldErrors) && !empty($fieldErrors)) {
                    return $fieldErrors[0]; // Return the first error message for the first field with errors.
                }
                if (is_string($fieldErrors)) {
                    return $fieldErrors; // Return the first error message.
                }
            }
        }

        return "Unknown error";
    }

    /**
     * @return string
     */
    public static function getUniversalErrorMessage(): string
    {
        return 'Something went wrong, please <a href="https://2index.ninja/account/tickets" target="_blank" title="Customer support">contact customer support<a/>';
    }

    /**
     * Api key must be at least 15 characters and contain only letters, number and special symbols
     * @param $apiKey
     * @return bool
     */
    public static function validateApiKey($apiKey): bool
    {
        return preg_match('/^[a-zA-Z0-9!@#$%^&*()_+-=|]{15,}$/', $apiKey);
    }

    /**
     * @param $response
     * @return bool
     */
    public static function isResponseValid($response): bool
    {
        if ($response === null && json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }
        return true;
    }
}