jQuery(document).ready(function ($) {
    /**
     * Set API key form show and focus
     */
    $('#index-set-api-key').click(function () {
        $('#api-key-form').show();
        $(this).hide();
        $('#index-set-api-key-text').hide();
        $('#index-api-key').focus();
    });

    /**
     * API key form submission
     */
    $('#api-key-form').on('submit', function (event) {
        event.preventDefault();

        const apiKeyInput = $('#index-api-key');
        const apiKey = apiKeyInput.val();
        var errorWrapper = $(this).find('.error-wrapper');
        var errorSpan = errorWrapper.find('.error-text');
        var submitButton = $(this).find('button[type="submit"]');

        if (!isValidApiKey(apiKey)) {
            errorSpan.text('API key must be at least 15 characters and contain letters, numbers and symbols.');
            errorSpan.show();
            errorWrapper.addClass('error');
        } else {
            errorSpan.text('');
            errorSpan.hide();
            errorWrapper.removeClass('error');

            submitButton.text('Loading...');
            submitButton.prop('disabled', true);

            $(this).unbind('submit').submit();
        }
    });

    /**
     * Check if API key is valid
     * @param apiKey
     * @returns {boolean}
     */
    function isValidApiKey(apiKey) {
        if (apiKey.length < 15) {
            return false;
        }
        return /^[a-zA-Z0-9!@#$%^&*()_+-=|]+$/.test(apiKey);
    }

    /**
     * Clear queue button click
     */
    $('#clear-queue-button').on("click", function (event) {
        event.preventDefault();

        let button = $(this);

        button.text('Erasing...');
        button.prop('disabled', true);

        var data = {
            action: 'index_ninja_clear_queue_action',
            security: TIDNAjax.security,
        };

        $.post(TIDNAjax.ajaxurl, data, function(response) {
            if (response.success) {
                showToast(response.data);
                $('#td-queue-count').text('0');
            } else {
                showToast(response.data, 'error');
            }
        }).then(function() {
            button.text('Clear queue');
            button.prop('disabled', false);
        });
    });

    /**
     * Toggle watch status button click
     */
    $('.button-toggle-watch-status').on("click", function(e) {
        let button = $(this);

        button.addClass('loading');

        var data = {
            action: 'index_ninja_toggle_watch_status_action',
            security: TIDNAjax.security,
            id: button.data('id'),
            watched: button.hasClass('watched'),
        };

        $.post(TIDNAjax.ajaxurl, data, function(response) {
            if (response.success) {
                showToast(response.data);
                button.toggleClass('watched');
            } else {
                showToast(response.data, 'error');
            }
        }).then(function() {
            button.removeClass('loading');
        });
    });

    /**
     * Remove sitemap button click
     */
    $('.button-remove-source').click(function (event) {
        event.preventDefault();

        let button = $(this);

        button.text('Deleting...');
        button.prop('disabled', true);

        var data = {
            action: 'index_ninja_delete_source_action',
            security: TIDNAjax.security,
            id: button.data('id'),
        };

        $.post(TIDNAjax.ajaxurl, data, function(response) {
            if (response.success) {
                showToast(response.data);
                $('#source-tr-' + button.data('id')).fadeOut();

                if ($('#sources-table tbody tr:visible').length === 1) {
                    $('#sources-table').fadeOut();
                    $('#no-sources-found').show();
                }
            } else {
                showToast(response.data, 'error');
            }
        }).then(function() {
            button.text('delete');
            button.prop('disabled', false);
        });
    });

    const engines = $('input[name="add-pages-form-engine[]"]');
    const sourceSelect = $('#add-pages-form-source');
    const sitemapSection = $('#sitemap-section');
    const listSection = $('#list-section');
    const fileSection = $('#file-section');
    const sitemapInput = $('#add-pages-form-sitemap-link');
    const listInput = $('#add-pages-form-list-links');
    const fileInput = $('#add-pages-form-file-input');

    /**
     * Validate selected engines
     * @returns {boolean}
     */
    function validateEngines() {
        let isChecked = false;
        engines.each(function () {
            if ($(this).is(':checked')) {
                isChecked = true;
            }
        });

        let engineError = $('.engines-wrapper .error-text');
        let errorWrapper = $('.engines-wrapper .error-wrapper');

        if (!isChecked) {
            engineError.text('*Please select at least one search engine');
            engineError.show();
            errorWrapper.addClass('error');
            return false;
        } else {
            engineError.text('');
            engineError.hide();
            errorWrapper.removeClass('error');
            return true;
        }
    }

    /**
     * Validate engines on checkbox changes
     */
    engines.on('change', function() {
        validateEngines();
    });

    /**
     * Validate inputs in sections
     * @returns {boolean}
     */
    function validateSectionsInputs() {
        let hasError = false;
        if (sourceSelect.val() === 'sitemap') {
            if (!sitemapInput.val() || sitemapInput.val().length === 0) {
                sitemapSection.find('.error-wrapper').addClass('error');
                sitemapSection.find('.error-text').show().text('Sitemap URL is required');
                hasError = true;
            } else {
                sitemapSection.find('.error-wrapper').removeClass('error');
                sitemapSection.find('.error-text').hide().text('');
            }
        } else if (sourceSelect.val() === 'list') {
            if (!listInput.val() || listInput.val().length === 0) {
                listSection.find('.error-wrapper').addClass('error');
                listSection.find('.error-text').show().text('List of URLs is required');
                hasError = true;
            } else {
                listSection.find('.error-wrapper').removeClass('error');
                listSection.find('.error-text').hide().text('');
            }
        } else if (sourceSelect.val() === 'file') {
            if (!fileInput.val() || fileInput.val().length === 0) {
                fileSection.find('.error-wrapper').addClass('error');
                fileSection.find('.error-text').show().text('File is required');
                hasError = true;
            } else {
                fileSection.find('.error-wrapper').removeClass('error');
                fileSection.find('.error-text').hide().text('');
            }
        }
        return !hasError;
    }

    /**
     * Reset errors in sections
     */
    function resetSectionsErrors() {
        sitemapSection.find('.error-wrapper').removeClass('error');
        sitemapSection.find('.error-text').hide().text('');

        listSection.find('.error-wrapper').removeClass('error');
        listSection.find('.error-text').hide().text('');

        fileSection.find('.error-wrapper').removeClass('error');
        fileSection.find('.error-text').hide().text('');
    }

    /**
     * Validate sections inputs on change
     */
    sourceSelect.on('change', function() {
        $('.form-section').hide();
        resetSectionsErrors();

        let selectedValue = $(this).val();
        if (selectedValue) {
            $('#' + selectedValue + '-section').show();
        }
    });

    /**
     * Validate sitemap input on change
     */
    fileInput.on('change', function() {
        if (this.files.length > 0) {
            const fileName = this.files[0].name;
            const label = $('label[for="add-pages-form-file-input"]');
            label.text(fileName).addClass('file-selected');
        } else {
            label.text('Select file').removeClass('file-selected');
        }
    });

    /**
     * Submit add pages form
     */
    $('#add-pages-form').on('submit', function (e) {
        e.preventDefault();

        if (!validateEngines() || !validateSectionsInputs()) {
            return;
        }

        const submitButtons = $(this).find('button[type="submit"]');
        submitButtons.each(function () {
            $(this).text('Sending...');
            $(this).prop('disabled', true);
        });

        $(this).unbind('submit').submit();
    });

    /**
     * Show toast message
     * @param message
     * @param type
     */
    function showToast(message, type = 'success') {
        let background = "linear-gradient(to right, rgb(0 176 80), rgb(42 131 69))";
        if (type === 'error') {
            background = "linear-gradient(to right, rgb(176 0 0), rgb(131 42 42))";
        }

        if (typeof Toastify !== 'function') {
            console.error('Toastify is not defined');
            return;
        }
        Toastify({
            text: message,
            close: true,
            position: 'center',
            style: {
                background: background,
            },
        }).showToast();
    }
});