<?php
/**
 * Plugin Name: Indexing website for Google
 * Plugin URI:  https://2index.ninja/wp-plugin
 * Description: This plugin submits your website pages to Google, Yandex and Bing for indexing using the 2index.ninja API.
 * Version:     1.0.21
 * Author:      2Index.ninja
 * Author URI:  https://2index.ninja
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: 2index-page-indexer
 * Domain Path: /languages/
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Function to load the plugin's textdomain.
function load_2index_page_indexer_textdomain() {
    load_plugin_textdomain( '2index-page-indexer', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
}
add_action( 'plugins_loaded', 'load_2index_page_indexer_textdomain' );

// On plugin init include the necessary files
function TIDN_init(): void
{
    include_once plugin_dir_path(__FILE__) . 'classes/api.php';
    include_once plugin_dir_path(__FILE__) . 'classes/helpers.php';
}

add_action('init', 'TIDN_init');

function TIDN_activate_plugin(): void
{
    // Code to be executed on plugin activation.
}

register_activation_hook(__FILE__, 'TIDN_activate_plugin');

function TIDN_deactivate_plugin(): void
{
    delete_option('TIDN_api_key');
}

register_deactivation_hook(__FILE__, 'TIDN_deactivate_plugin');

function TIDN_menu(): void
{
    add_menu_page(
        esc_html__('Page Indexer Plugin', '2index-page-indexer'),
        esc_html__('Page Indexer', '2index-page-indexer'),
        'manage_options',
        'index-ninja-index',
        'TIDN_index_page',
        'dashicons-search',
        60
    );

    add_submenu_page(
        'index-ninja-index',
        esc_html__('Dashboard', '2index-page-indexer'),
        esc_html__('Dashboard', '2index-page-indexer'),
        'manage_options',
        'index-ninja-index',
        'TIDN_index_page'
    );

    add_submenu_page(
        'index-ninja-index',
        esc_html__('Indexing Pages', '2index-page-indexer'),
        esc_html__('Indexing Pages', '2index-page-indexer'),
        'manage_options',
        'index-ninja-indexing-pages',
        'TIDN_indexing_pages_page'
    );

    add_submenu_page(
        'index-ninja-index',
        esc_html__('Settings', '2index-page-indexer'),
        esc_html__('Settings', '2index-page-indexer'),
        'manage_options',
        'index-ninja-settings',
        'TIDN_settings_page'
    );

    add_submenu_page(
        'index-ninja-index',
        esc_html__('FAQ', '2index-page-indexer'),
        esc_html__('FAQ', '2index-page-indexer'),
        'manage_options',
        'index-ninja-faq',
        'TIDN_faq_page'
    );
}

add_action('admin_menu', 'TIDN_menu');

/**
 * Enqueues scripts and styles for the plugin.
 * @return void
 */
function TIDN_enqueue_scripts($hook): void
{
    // Only enqueue scripts and styles on plugin pages
    if (!strpos($hook, 'index-ninja')) {
        return;
    }

    // Enqueue CSS
    wp_enqueue_style('TIDN-style', plugins_url('assets/css/style.css', __FILE__));
    wp_enqueue_style('TIDN-toastify-style', plugins_url('assets/toastify/toastify.min.css', __FILE__), array(), '1.12.0');

    // Enqueue JS (with jQuery dependency)
    wp_enqueue_script('TIDN-script', plugins_url('assets/js/script.js', __FILE__), array('jquery'), '1.0.0', true);
    wp_enqueue_script('TIDN-toastify-script', plugins_url('assets/toastify/toastify.min.js', __FILE__), array(), '1.12.0', true);

    // Localize script with the AJAX URL
    wp_localize_script('TIDN-script', 'TIDNAjax', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'security' => wp_create_nonce('TIDN_nonce')
    ));
}

add_action('admin_enqueue_scripts', 'TIDN_enqueue_scripts');

/**
 * Handles the AJAX request Clear Queue for the plugin.
 * @return void
 */
function TIDN_clear_queue_action_ajax_callback()
{
    check_ajax_referer('TIDN_nonce', 'security');

    $apiKey = TIDN_Helpers::getApiKey();
    $api = new TIDN_Api($apiKey);

    $project = $api->get_wordpress_project(trailingslashit(get_site_url()));

    if (TIDN_Helpers::isQueryError($project)) {
        wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($project));
        wp_die();
    }

    if (TIDN_Helpers::isQuerySuccess($project)) {
        $project = $project['project'];
        $response = $api->clear_queue($project['id']);

        if (TIDN_Helpers::isQuerySuccess($response)) {
            wp_send_json_success(TIDN_Helpers::getQueryMessage($response));
        } else {
            wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($response));
        }

        wp_die();
    }

    wp_send_json_error(esc_html__('Unknown error occurred.', '2index-page-indexer'));
    wp_die();
}

add_action('wp_ajax_index_ninja_clear_queue_action', 'TIDN_clear_queue_action_ajax_callback');

/**
 * Handles the AJAX request Toggle Watch Status for the plugin.
 * @return void
 */
function TIDN_toggle_watch_status_action_ajax_callback()
{
    check_ajax_referer('TIDN_nonce', 'security');

    if (!isset($_POST['id'], $_POST['watched'])) {
        wp_send_json_error(esc_html__('ID and watch status are required.', '2index-page-indexer'));
        wp_die();
    }

    $sourceId = absint($_POST['id']); // Sanitize ID
    $newStatus = sanitize_text_field($_POST['watched']) !== 'true'; // Sanitize and convert to boolean

    $apiKey = TIDN_Helpers::getApiKey();
    $api = new TIDN_Api($apiKey);

    $project = $api->get_wordpress_project(trailingslashit(get_site_url()));

    if (TIDN_Helpers::isQueryError($project)) {
        wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($project));
        wp_die();
    }

    if (TIDN_Helpers::isQuerySuccess($project)) {
        $project = $project['project']; // No need for extra conditional

        $response = $api->update_sitemap_watch($project['id'], $sourceId, $newStatus);

        if (TIDN_Helpers::isQuerySuccess($response)) {
            wp_send_json_success(esc_html__('Watch status updated.', '2index-page-indexer'));
        } else {
            wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($response));
        }

        wp_die();
    }

    wp_send_json_error(esc_html__('Unknown error occurred.', '2index-page-indexer'));
    wp_die();
}

add_action('wp_ajax_index_ninja_toggle_watch_status_action', 'TIDN_toggle_watch_status_action_ajax_callback');

/**
 * Handles the AJAX request Delete Source for the plugin.
 * @return void
 */
function TIDN_delete_source_action_ajax_callback()
{
    check_ajax_referer('TIDN_nonce', 'security');

    if (!isset($_POST['id']) || !is_numeric($_POST['id'])) {
        wp_send_json_error(esc_html__('Invalid request data.', '2index-page-indexer'));
        wp_die();
    }

    $sourceId = absint($_POST['id']);

    $apiKey = TIDN_Helpers::getApiKey();
    $api = new TIDN_Api($apiKey);

    $project = $api->get_wordpress_project(trailingslashit(get_site_url()));

    if (TIDN_Helpers::isQueryError($project)) {
        wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($project));
        wp_die();
    }

    if (TIDN_Helpers::isQuerySuccess($project)) {
        $project = $project['project']; // No need for extra if statement

        $response = $api->delete_sitemap($project['id'], $sourceId);

        if (TIDN_Helpers::isQuerySuccess($response)) {
            wp_send_json_success(TIDN_Helpers::getQueryMessage($response));
        } else {
            wp_send_json_error(TIDN_Helpers::getQueryErrorMessage($response));
        }

        wp_die();
    }

    wp_send_json_error(esc_html__('Unknown error occurred.', '2index-page-indexer'));
    wp_die();
}

add_action('wp_ajax_index_ninja_delete_source_action', 'TIDN_delete_source_action_ajax_callback');

/**
 * Displays the content of the "Index" page.
 *
 * @return void
 */
function TIDN_index_page(): void
{
    include_once(plugin_dir_path(__FILE__) . 'templates/index.php');
}

/**
 * Displays the content of the "Indexing Pages" page.
 *
 * @return void
 */
function TIDN_indexing_pages_page(): void
{
    $apiKey = TIDN_Helpers::getApiKey();

    if (!$apiKey || !TIDN_Helpers::isApiKeyRegistered($apiKey)) {
        add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Please enter the API key in the settings.', '2index-page-indexer'), 'error');
        include_once plugin_dir_path(__FILE__) . 'templates/settings/index.php';

        return;
    }

    if (!TIDN_Helpers::isApiKeyVerified($apiKey)) {
        include_once plugin_dir_path(__FILE__) . 'templates/settings/index.php';

        return;
    }

    $action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : '';

    if ($action === '') {
        include_once plugin_dir_path(__FILE__) . 'templates/indexing/index.php';

        return;
    }

    if ($action === 'add-pages') {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $engines = isset($_POST['add-pages-form-engine']) ? array_map('sanitize_text_field', $_POST['add-pages-form-engine']) : []; // Sanitize engines array
            $source = isset($_POST['add-pages-form-source']) ? sanitize_text_field($_POST['add-pages-form-source']) : '';     // Sanitize source

            if (empty($engines)) {
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('At least one search engine must be selected.', '2index-page-indexer'), 'error');
                include_once plugin_dir_path(__FILE__) . 'templates/indexing/add-pages.php';

                return;
            }

            if (empty($source)) {
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Source of pages is required.', '2index-page-indexer'), 'error');
                include_once plugin_dir_path(__FILE__) . 'templates/indexing/add-pages.php';

                return;
            }

            $api = new TIDN_Api($apiKey);
            $project = $api->get_wordpress_project(trailingslashit(get_site_url()));

            if (TIDN_Helpers::isQueryError($project)) {
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', TIDN_Helpers::getQueryErrorMessage($project), 'error');
                include_once plugin_dir_path(__FILE__) . 'templates/indexing/add-pages.php';

                return;
            }

            if (TIDN_Helpers::isQuerySuccess($project)) {
                $project = $project['project'];

                $setGoogle = in_array('google', $engines);
                $setYandex = in_array('yandex', $engines);
                $setBing = in_array('bing', $engines);
                $response = null;

                switch ($source) {
                    case 'sitemap':
                        $link = isset($_POST['add-pages-form-sitemap-link']) ? esc_url_raw($_POST['add-pages-form-sitemap-link']) : '';


                        if (empty($link)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Sitemap link is required.', '2index-page-indexer'), 'error');
                            break;
                        }

                        if (!filter_var($link, FILTER_VALIDATE_URL)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Sitemap link is not valid.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $submitAllNewPages = isset($_POST['submit-all-new-pages']);

                        $response = $api->add_sitemap($project['id'], $link, $setGoogle, $setYandex, $setBing, null, $submitAllNewPages);

                        break;
                    case 'list':
                        if (!empty($_POST['add-pages-form-list-links'])) {
                            $linksList = sanitize_textarea_field($_POST['add-pages-form-list-links']);
                            $links = explode("\n", $linksList);
                        } else {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('List of links is required.', '2index-page-indexer'), 'error');
                            break;
                        }

                        if (count($links) > 1000) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File must contain less than 1000 links.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $links = array_filter($links, function ($link) {
                            return filter_var(trim($link), FILTER_VALIDATE_URL);
                        });

                        if (empty($links)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('No valid links found in the list.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $links = implode("\n", $links);
                        $response = $api->add_links($project['id'], $links, $setGoogle, $setYandex, $setBing);
                        break;

                    case 'file':
                        if (!isset($_FILES['add-pages-form-file-input'])) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File upload failed. No file provided.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $file = $_FILES['add-pages-form-file-input'];

                        $fileName = sanitize_file_name($file['name']);
                        $fileTmpName = $file['tmp_name']; //Using realpath() is unnecessary and potentially insecure
                        $fileSize = filesize($fileTmpName);

                        if (!is_uploaded_file($fileTmpName)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File upload failed.', '2index-page-indexer'), 'error');
                            break;
                        }

                        if ($fileSize === false) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Error reading the file.', '2index-page-indexer'), 'error');
                            break;
                        }

                        if ($file['error'] !== UPLOAD_ERR_OK) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File upload failed.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $allowed_mime_types = array('text/plain');
                        $fileInfo = finfo_open(FILEINFO_MIME_TYPE);
                        $mime_type = finfo_file($fileInfo, $fileTmpName);
                        finfo_close($fileInfo);

                        if (!in_array($mime_type, $allowed_mime_types)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Invalid file type. Only plain text files are allowed.', '2index-page-indexer'), 'error');
                            break;
                        }

                        if ($fileSize > 52428800) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File size must be less than 50MB.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $fileContent = file_get_contents($fileTmpName);
                        if ($fileContent === false) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('Error reading the file.', '2index-page-indexer'), 'error');
                            break;
                        }
                        $links = explode("\n", $fileContent);

                        if (count($links) > 100000) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('File must contain less than 100,000 links.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $links = array_map('esc_url_raw', array_map('trim', $links));
                        $links = array_filter($links, 'wp_http_validate_url');

                        if (empty($links)) {
                            add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('No valid links found in the file.', '2index-page-indexer'), 'error');
                            break;
                        }

                        $links = implode("\n", $links);
                        $response = $api->add_links($project['id'], $links, $setGoogle, $setYandex, $setBing);
                        break;
                }

                if ($response !== null) {
                    if (TIDN_Helpers::isQuerySuccess($response)) {
                        add_settings_error('index_ninja_api_key', 'index_ninja_api_key_success', TIDN_Helpers::getQueryMessage($response), 'success');
                    } else {
                        $invalidLinksText = TIDN_Helpers::getInvalidLinksText($response);
                        add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', $invalidLinksText ?: TIDN_Helpers::getQueryErrorMessage($response), 'error');
                    }
                }
            }
        }

        include_once plugin_dir_path(__FILE__) . 'templates/indexing/add-pages.php';
        return;
    }
}

/**
 * Displays the content of the "Settings" page.
 *
 * @return void
 */
function TIDN_settings_page(): void
{
    $apiKey = TIDN_Helpers::getApiKey();

    if ($apiKey) {
        TIDN_Helpers::isApiKeyRegistered($apiKey);
    }

    if (isset($_POST['action'])) {
        $action = sanitize_text_field($_POST['action']);

        if ($action === 'add-api-key') {
            if (!isset($_POST['index-api-key'])) {
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('API key is required.', '2index-page-indexer'), 'error');
            } else {
                $apiKey = sanitize_text_field($_POST['index-api-key']);

                if (!TIDN_Helpers::validateApiKey($apiKey)) {
                    add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('API key must be at least 15 characters and contain only letters and numbers.', '2index-page-indexer'), 'error');
                }

                if (!TIDN_Helpers::isApiKeyRegistered($apiKey)) {
                    add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('API key is not registered.', '2index-page-indexer'), 'error');
                }

                TIDN_Helpers::setApiKey($apiKey);
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_success', esc_html__('API key saved.', '2index-page-indexer'), 'success');
            }
        } elseif ($action === 'register-user') {
            $api = new TIDN_Api();
            $admin_email = sanitize_email(get_option('admin_email'));
            $response = $api->register_user($admin_email);

            if (TIDN_Helpers::isQuerySuccess($response)) {
                if (TIDN_Helpers::setApiKeyFromResponse($response)) {
                    add_settings_error('index_ninja_api_key', 'index_ninja_api_key_success', esc_html__('You have been successfully registered.', '2index-page-indexer'), 'success');
                } else {
                    add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', esc_html__('API key not found in the response.', '2index-page-indexer'), 'error');
                }
            } else {
                add_settings_error('index_ninja_api_key', 'index_ninja_api_key_error', TIDN_Helpers::getFirstError($response), 'error');
            }
        }
    }

    include_once plugin_dir_path(__FILE__) . 'templates/settings/index.php';
}

/**
 * Displays the content of the "FAQ" page.
 *
 * @return void
 */
function TIDN_faq_page(): void
{
    include_once(plugin_dir_path(__FILE__) . 'templates/faq.php');
}

/**
 * @return string
 */
function TIDN_get_plugin_dir_url(): string
{
    return plugin_dir_url(__FILE__);
}